/**
 * \file sdc_op_digest .c
 *
 * \brief Functions for digest
 * Please note : the implementation is split into different operations
 * instead of splitting it in common, convenience and advanced functions
 *
 * \author Christoph Gellner (cgellner@de.adit-jv.com)
 *
 * \copyright (c) 2017 Advanced Driver Information Technology.
 * This code is developed by Advanced Driver Information Technology.
 * Copyright of Advanced Driver Information Technology, Bosch, and DENSO.
 * All rights reserved.
 *
 *
 ***********************************************************************/

#include <stdlib.h>
#include <string.h>
#include <sdc_op_common.h>
#include <sdc_op_adv.h>
#include <sdc_op_conv.h>
#include <sdc_random.h>
#include <private/sdc_arch.h>
#include <private/sdc_intern.h>

/* Definitions types and defaults */
static const char *sdc_dgst_hash_names[] = {
    [SDC_DGST_HASH_MD5] = "MD5",
    [SDC_DGST_HASH_SHA1] = "SHA1",
    [SDC_DGST_HASH_SHA224] = "SHA224",
    [SDC_DGST_HASH_SHA256] = "SHA256",
    [SDC_DGST_HASH_SHA384] = "SHA384",
    [SDC_DGST_HASH_SHA512] = "SHA512"
};

/* Functions */

/**
 * \brief Checks common to digest
 *
 * Check session, type and in data
 * Provide \ref sdc_dgst_desc_t of type
 */
static sdc_error_t sdc_dgst_common_checks_defaults(
    sdc_error_t error_init,
    sdc_session_t *session,
    const sdc_dgst_type_t *type,
    sdc_dgst_desc_t *internal_desc,
    const uint8_t *in_data, const size_t in_len)
{

    sdc_error_t err = error_init;

    if (SDC_OK != sdc_intern_check_data_input_buffer(in_data, in_len))
        err = SDC_IN_DATA_INVALID;

    if (!type)
        err = SDC_ALG_MODE_INVALID;

    if (err == SDC_OK)
        err = sdc_dgst_desc_fill (type, internal_desc);

    if (!session)
        err = SDC_SESSION_INVALID;

    return err;
}

static sdc_error_t sdc_dgst_common_check_digest(const sdc_dgst_desc_t *desc, size_t digest_len)
{
    if (SDC_OK != sdc_intern_range_min_max_mod_check(digest_len, &(desc->digest)))
        return SDC_DGST_DATA_INVALID;

    return SDC_OK;
}

/**
 * \brief call min max checks for in data and digest
 */
static sdc_error_t sdc_dgst_common_check_digest_in_length(
    const sdc_dgst_desc_t *desc,
    size_t in_len,
    size_t digest_len
    )
{
    sdc_error_t err;

    err = sdc_intern_input_only_check_min_max_align(in_len,
                                                    &(desc->data));
    if (err == SDC_OK)
        err = sdc_dgst_common_check_digest(desc, digest_len);

    return err;
}

static sdc_error_t sdc_session_validation(sdc_session_t *session)
{
    /* verify session */
    if (!session)
        return SDC_SESSION_INVALID;

    return SDC_OK;
}

static sdc_error_t sdc_type_desc_validation(const sdc_dgst_type_t *type,
                                            const sdc_dgst_desc_t *desc)
{
    /* verify type and desc */
    if(!type)
        return SDC_ALG_MODE_INVALID;
    if(!desc)
        return SDC_INVALID_PARAMETER;

    return SDC_OK;
}

static sdc_error_t sdc_session_type_desc_validation(sdc_session_t *session,
                                              const sdc_dgst_type_t *type,
                                              const sdc_dgst_desc_t *desc)
{
    sdc_error_t err = SDC_OK;
    /* verify inputs */
    if ((err = sdc_session_validation(session)) != SDC_OK)
        return err;
    if((err = sdc_type_desc_validation(type, desc)) != SDC_OK)
        return err;

    return err;
}

/* check digest parameters */
static sdc_error_t sdc_dgst_check_digest_data (uint8_t **digest_data, size_t *digest_len)
{
    if ((digest_data == NULL) || (digest_len == NULL))
        return SDC_DGST_DATA_INVALID;

    return SDC_OK;
}

static sdc_error_t sdc_common_dgst_init(sdc_session_t *session,
                                        const sdc_dgst_type_t *type,
                                        const sdc_dgst_desc_t *desc)
{
    session->unaligned_buffer_filllevel = 0;
    session->inlen_cnt = 0;

    if (!desc->supports_iuf)
        return SDC_OP_NOT_SUPPORTED;

    return sdc_arch_dgst_init(session, type, desc);
}

static sdc_error_t sdc_common_dgst_update(sdc_session_t *session,
                                          const sdc_dgst_type_t *type,
                                          const sdc_dgst_desc_t *desc,
                                          const uint8_t *in_data, const size_t in_data_len)
{
    sdc_error_t err = SDC_OK;
    size_t block_len = desc->data.block_len;
    size_t chunk_len = desc->data.max_chunk_len;
    bool chunk_len_aligned = desc->data.chunk_len_aligned;
    const uint8_t *next_in_data;
    size_t in_len_remaining;
    size_t unaligned_buf_rem;
    uint8_t *unaligned_buf_next;
    size_t min_no_proc_len;
    size_t process_in_len;
    size_t remainder;

    if (session->unaligned_buffer_filllevel > block_len)
        err = SDC_INTERNAL_ERROR;

    /* Check that processing additional data won't exceed max */
    if (err == SDC_OK)
        err = sdc_intern_input_only_check_update_wont_exceed_max(
                session->inlen_cnt,
                in_data_len,
                &(desc->data));

    next_in_data = in_data;
    in_len_remaining = in_data_len;

    /* handle previous unaligned data first */
    if ((err == SDC_OK) && (session->unaligned_buffer_filllevel > 0)) {
        /* if unaligned_buffer_filllevel > 0 we obviously need to alignment */
        unaligned_buf_rem = block_len - session->unaligned_buffer_filllevel;

        if (!chunk_len_aligned) {
            /*
             * This must not happen
             * Probably someone has corrupted the control structures
             */
            err = SDC_INTERNAL_ERROR;
        } else {
            if (in_len_remaining >= unaligned_buf_rem)
            {
                unaligned_buf_next = session->unaligned_buffer;
                unaligned_buf_next += session->unaligned_buffer_filllevel;

                /* afterwards we have one block in the buffer */
                memcpy(unaligned_buf_next, next_in_data, unaligned_buf_rem);
                /* process the unaligned part */
                err = sdc_arch_dgst_update(session, type, desc,
                                           session->unaligned_buffer, block_len);
                if (err == SDC_OK) {
                    /* update pointers + lengths for remaining data */
                    next_in_data += unaligned_buf_rem;
                    in_len_remaining -= unaligned_buf_rem;

                    /* all data handled */
                    session->unaligned_buffer_filllevel = 0;
                }
            }
            /*
             * else case
             *       i.e. in_len_remaining < unaligned_buf_mem
             * will be handled by "append remaining data to unaligned_buffer"
             *
             * In this case
             *      in_len_remaining is < block_len as well and chunk_len_aligned true.
             *
             * Otherwise there is no way (beside fooling around with internal
             * structs) that data has been added to unaligned_buffer in the
             * previous call.
             *
             * Note: the while loop won't process any data in this case
             */
        }
    }

    /* smaller or equal length need to be handled using unaligned_buffer */
    min_no_proc_len = 0;
    if (chunk_len_aligned)
        min_no_proc_len = block_len - 1;

    while ((err == SDC_OK) && (in_len_remaining > min_no_proc_len)) {
        process_in_len = in_len_remaining;

        if (process_in_len > chunk_len)
            process_in_len = chunk_len;

        if (chunk_len_aligned) {
            /* align */
            remainder = (process_in_len % block_len);
            if (remainder != 0) {
                process_in_len -= remainder;
            }
        }
        /* process the next chunk of data */
        err = sdc_arch_dgst_update(session, type, desc,
                                   next_in_data, process_in_len);

        if (err == SDC_OK) {
            /* update pointers + lengths for remaining data */
            next_in_data += process_in_len;
            in_len_remaining -= process_in_len;
        }
    }

    /* append remaining data to unaligned_buffer */
    if ((err == SDC_OK) && (in_len_remaining > 0)) {
        unaligned_buf_rem = block_len - session->unaligned_buffer_filllevel;

        if (in_len_remaining > unaligned_buf_rem) {
            /* this must not happen */
            err = SDC_INTERNAL_ERROR;
        } else {
            unaligned_buf_next = session->unaligned_buffer;
            size_t fill_level = session->unaligned_buffer_filllevel;
            unaligned_buf_next += fill_level;

            /* append to the end of the unaligned buffer */
            memcpy(unaligned_buf_next, next_in_data, in_len_remaining);

            session->unaligned_buffer_filllevel += in_len_remaining;

            /* no need to update in_data_remaining */
            in_len_remaining = 0;
        }
    }

    if (err != SDC_OK) {
        /* clear confidential data in case of error */
        sdc_intern_clear_session_confidential(session);
    } else {
        /* add the current input data length */
        session->inlen_cnt += in_data_len;
    }

    return err;
}

static sdc_error_t sdc_common_dgst_finalize(sdc_session_t *session,
                                            const sdc_dgst_type_t *type,
                                            const sdc_dgst_desc_t *desc,
                                            uint8_t *digest_out_data, size_t digest_out_data_len)
{
    sdc_error_t err = SDC_OK;
    size_t unaligned_data_len;
    sdc_padding_t padding = desc->data.padding;
    size_t block_len = desc->data.block_len;
    bool padded = false;

    unaligned_data_len = session->unaligned_buffer_filllevel;

    /* Check that total provided length fits to min/max */
    err = sdc_intern_input_only_check_min_max_align(
            session->inlen_cnt,
            &(desc->data));

    if ((padding != SDC_PADDING_INTERNAL) &&
        (padding != SDC_PADDING_NO) &&
        (padding != SDC_PADDING_HIDDEN))
        padded = true;

    if ((err == SDC_OK) && (padded)) {
        err = sdc_intern_pad(padding,
                             session->unaligned_buffer,
                             unaligned_data_len,
                             block_len);
        unaligned_data_len = block_len;
    }
    if (err == SDC_OK) {
        err = sdc_arch_dgst_finalize(session, type, desc,
                                     session->unaligned_buffer,
                                     unaligned_data_len,
                                     digest_out_data, digest_out_data_len);
    }

    sdc_intern_clear_session_confidential(session);
    return err;
}

/**
 * \brief common implementation of digest using architecture dependent
 * init, update, finalize functions
 *
 * This function will be used when the architecture dependent part does not provide
 * optimized digest functionality
 *
 * \todo add functionality
 *
 */
static sdc_error_t sdc_common_dgst(sdc_session_t *session,
                                   const sdc_dgst_type_t *type,
                                   const sdc_dgst_desc_t *desc,
                                   const uint8_t *in_data,
                                   const size_t in_data_len,
                                   uint8_t *digest_out_data, const size_t digest_out_len)
{
    sdc_error_t err;
    err = sdc_common_dgst_init(session, type, desc);

    if (err == SDC_OK) {
        err = sdc_common_dgst_update(session, type, desc, in_data, in_data_len);
    }

    if (err == SDC_OK) {
        err = sdc_common_dgst_finalize(session, type, desc, digest_out_data, digest_out_len);
    }

    return err;
}

/**
 * \brief select if architecture specific or common version is used
 */
static sdc_error_t sdc_dgst_selector(sdc_session_t *session,
                                     const sdc_dgst_type_t *type,
                                     const sdc_dgst_desc_t *desc,
                                     const uint8_t *in_data,
                                     const size_t in_data_len,
                                     uint8_t *digest_out_data, const size_t digest_out_len)
{

    sdc_error_t err;

    err = sdc_arch_dgst(session,
                        type,
                        desc,
                        in_data,
                        in_data_len,
                        digest_out_data, digest_out_len);

    if (err == SDC_NOT_SUPPORTED) {
        err = sdc_common_dgst(session,
                              type,
                              desc,
                              in_data,
                              in_data_len,
                              digest_out_data, digest_out_len);
    }

    return err;
}

sdc_error_t sdc_dgst(sdc_session_t *session,
                     const sdc_dgst_type_t *type,
                     const uint8_t *in_data, const size_t in_len,
                     uint8_t **digest_data, size_t *digest_len)
{

    sdc_error_t err = SDC_OK;
    sdc_error_t tmp_err;
    sdc_dgst_desc_t internal_desc;

    uint8_t *internal_digest = NULL;
    size_t internal_digest_len;

    /* initialize - in case failing with error we want to return
     * buffer pointer NULL and buffer len 0 for all not explicitly specified buffers
     */
    if (SDC_OK != sdc_intern_check_init_dgst_tag_iv_output_buffer(digest_data, digest_len, &internal_digest_len, SDC_DGST_USE_DEFAULT, NULL)) {
        err = SDC_DGST_DATA_INVALID;
    }

    err = sdc_dgst_common_checks_defaults(err,
                                          session,
                                          type, &internal_desc,
                                          in_data, in_len);

    if (err != SDC_OK)
        return err;

    if (internal_digest_len == SDC_DGST_USE_DEFAULT) {
        internal_digest_len = internal_desc.digest.dflt;
    }

    err = sdc_dgst_common_check_digest_in_length(&internal_desc, in_len, internal_digest_len);

    if (err == SDC_OK) {
        if (internal_digest_len != 0) {
            internal_digest = malloc(internal_digest_len);
            if (!internal_digest)
                err = SDC_NO_MEM;
        }
    }

    if (err == SDC_OK) {
        err = sdc_dgst_selector(session,
                                type,
                                &internal_desc,
                                in_data,
                                in_len,
                                internal_digest, internal_digest_len);
    }

    if (err == SDC_OK) {
        *digest_data = internal_digest;
        *digest_len = internal_digest_len;
    } else {
        /* clean allocated memory */
        free(internal_digest);
    }

    if (err == SDC_OK) {
        err = sdc_intern_ops_sequence_ctl_set(session, SDC_OP_NO_CRYPTO, SDC_NO_OP);
    } else {
        /* If current operation is failed, reset the history of "operation" and "function" parameters
         * since doesn't support retry operation */
        tmp_err = sdc_intern_ops_sequence_ctl_set(session, SDC_OP_DGST, SDC_OP_ERROR);
        if (tmp_err != SDC_OK)
            err = tmp_err;
    }

    return err;
}

/* defined in sdc_op_common.h */
const sdc_dgst_type_t *sdc_dgst_get_default(void)
{
    return sdc_arch_dgst_get_default();
}

/* defined in sdc_op_common.h */
sdc_error_t sdc_dgst_type_alloc(sdc_dgst_type_t **type)
{
    if (!type)
        return SDC_ALG_MODE_INVALID;

    return sdc_arch_dgst_type_alloc(type);
}

/* defined in sdc_op_common.h */
sdc_error_t sdc_dgst_type_free(sdc_dgst_type_t *type)
{
    if (!type)
        return SDC_ALG_MODE_INVALID;

    return sdc_arch_dgst_type_free(type);
}

/* defined in sdc_op_common.h */
sdc_error_t sdc_dgst_type_set_hash(sdc_dgst_type_t *type, sdc_dgst_hash_t hash)
{
    if (!type)
        return SDC_ALG_MODE_INVALID;

    if ((hash < SDC_DGST_HASH_FIRST) || (hash >= SDC_DGST_HASH_END))
        return SDC_ALG_MODE_INVALID;

    return sdc_arch_dgst_type_set_hash(type, hash);
}

/* defined in sdc_op_common.h */
sdc_error_t sdc_dgst_type_set_opt_bmsk(sdc_dgst_type_t *type, uint64_t opt_bmsk)
{
    if (!type)
        return SDC_ALG_MODE_INVALID;

    if (opt_bmsk != 0) /* nothing supported so far */
        return SDC_OP_NOT_SUPPORTED;

    /* as we don't support any option at the moment, we don't need any arch interface */
    return SDC_OK;
}

/* defined in sdc_op_common.h */
sdc_error_t sdc_dgst_type_get_hash(const sdc_dgst_type_t *type, sdc_dgst_hash_t *hash)
{
    if (!type)
        return SDC_ALG_MODE_INVALID;

    if (!hash)
        return SDC_INVALID_PARAMETER;

    return sdc_arch_dgst_type_get_hash(type, hash);
}

/* defined in sdc_op_common.h */
sdc_error_t sdc_dgst_type_get_opt_bmsk(const sdc_dgst_type_t *type, uint64_t *opt_bmsk)
{
    if (!type)
        return SDC_ALG_MODE_INVALID;

    if (!opt_bmsk)
        return SDC_INVALID_PARAMETER;

    /* as we don't support any option at the moment, we don't need any arch interface - simply return 0*/
    *opt_bmsk = 0;
    return SDC_OK;
}

/* defined in sdc_op_common.h */
sdc_error_t sdc_dgst_desc_alloc(sdc_dgst_desc_t **desc)
{
    if (desc == NULL) {
        return SDC_INVALID_PARAMETER;
    }

    *desc = malloc(sizeof(sdc_dgst_desc_t));
    if (*desc == NULL) {
        return SDC_NO_MEM;
    }

    memset(*desc, 0, sizeof(sdc_dgst_desc_t));

    return SDC_OK;
}

/* defined in sdc_op_common.h */
sdc_error_t sdc_dgst_desc_free(sdc_dgst_desc_t *desc)
{
    if (desc == NULL) {
        return SDC_INVALID_PARAMETER;
    }

    free(desc);

    return SDC_OK;
}

/* defined in sdc_op_common.h */
sdc_error_t sdc_dgst_desc_fill (
    const sdc_dgst_type_t *type,
    sdc_dgst_desc_t *desc)
{
    if (!type)
        return SDC_ALG_MODE_INVALID;

    if (!desc)
        return SDC_INVALID_PARAMETER;

    return sdc_arch_dgst_desc_fill(type, desc);
}

/* defined in sdc_op_common.h */
sdc_error_t sdc_dgst_desc_get_digest(sdc_dgst_desc_t *desc,
                                     size_t *min_val,
                                     size_t *max_val,
                                     size_t *mod,
                                     size_t *default_val)
{
    if (!desc)
        return SDC_INVALID_PARAMETER;

    return sdc_intern_range_get_min_max_mod_dflt(&(desc->digest),
                                           min_val,
                                           max_val,
                                           mod,
                                           default_val);
}

const char* sdc_dgst_hash_name (sdc_dgst_hash_t hash)
{
    size_t idx;
    size_t elems;

    if ((hash < SDC_DGST_HASH_FIRST) || (hash >= SDC_DGST_HASH_END))
        return NULL;

    idx = hash;
    elems = sizeof(sdc_dgst_hash_names) / sizeof(const char *);

    if (idx<elems) {
        return sdc_dgst_hash_names[idx];
    }

    return NULL;
}

sdc_error_t sdc_dgst_desc_get_max_chunk_len(const sdc_dgst_desc_t *desc, size_t *max_val)
{
    size_t max_chunk;

    if (!desc)
        return SDC_INVALID_PARAMETER;

    if (max_val) {
        max_chunk = desc->data.max_chunk_len;
        if (desc->data.chunk_len_aligned) {
            max_chunk -= (max_chunk % desc->data.block_len);
        }

        if (max_chunk == 0)
            return SDC_INTERNAL_ERROR;

        *max_val = max_chunk;
    }

    return SDC_OK;
}

sdc_error_t sdc_dgst_init(sdc_session_t *session,
                          const sdc_dgst_type_t *type,
                          const sdc_dgst_desc_t *desc)
{
    sdc_error_t err = SDC_OK;
    sdc_error_t tmp_err;

    /* verify session */
    err = sdc_session_validation(session);
    if(err == SDC_OK)
        err = sdc_intern_ops_sequence_ctl_check(session, SDC_OP_DGST, SDC_NO_OP);

    if(err == SDC_OK)
        err = sdc_type_desc_validation(type, desc);

    if (err == SDC_OK)
        err = sdc_common_dgst_init(session, type, desc);

    if(err == SDC_OK) {
        err = sdc_intern_ops_sequence_ctl_set(session, SDC_OP_DGST, SDC_OP_INITILIZED);
    } else {
        /* If current operation is failed, reset the history of "operation" and "function" parameters
        * since doesn't support retry operation */
        tmp_err = sdc_intern_ops_sequence_ctl_set(session, SDC_OP_DGST, SDC_OP_ERROR);
        if(tmp_err != SDC_OK)
            err = tmp_err;
    }

    return err;
}

sdc_error_t sdc_dgst_update(sdc_session_t *session,
                            const sdc_dgst_type_t *type,
                            const sdc_dgst_desc_t *desc,
                            const uint8_t *in_data, const size_t in_len)
{
    sdc_error_t err = SDC_OK;
    sdc_error_t tmp_err;

    /* verify session */
    err = sdc_session_validation(session);
    if(err == SDC_OK)
        err = sdc_intern_ops_sequence_ctl_check(session, SDC_OP_DGST, SDC_OP_INITILIZED);

    if(err == SDC_OK)
        err = sdc_type_desc_validation(type, desc);

    if (err == SDC_OK) {
        if (sdc_intern_check_data_input_buffer(in_data, in_len) != SDC_OK)
            err = SDC_IN_DATA_INVALID;
    }
    if(err == SDC_OK)
        err = sdc_common_dgst_update(session, type, desc, in_data, in_len);

    /* No need to update sequence statues if operation is success since INIT
       * already had done status change which is enough for farther operations(FINALIZE).
       *
       * If current operation is failed, reset the history of "operation" and "function" parameters
       * since doesn't support retry operation
       */
    if(err != SDC_OK) {
        tmp_err = sdc_intern_ops_sequence_ctl_set(session, SDC_OP_DGST, SDC_OP_ERROR);
        if(tmp_err != SDC_OK)
            err = tmp_err;
    }

    return err;
}

sdc_error_t sdc_dgst_finalize(sdc_session_t *session,
                              const sdc_dgst_type_t *type,
                              const sdc_dgst_desc_t *desc,
                              uint8_t **digest_data, size_t *digest_len)
{
    sdc_error_t err = SDC_OK;
    sdc_error_t tmp_err;
    uint8_t *internal_digest =  NULL;
    size_t internal_digest_len = 0;

    /* verify inputs */
    err = sdc_session_type_desc_validation(session, type, desc);
    if(err == SDC_OK)
        err = sdc_intern_ops_sequence_ctl_check(session, SDC_OP_DGST, SDC_OP_INITILIZED);

    if(err == SDC_OK)
        err = sdc_dgst_check_digest_data(digest_data, digest_len);

    if(err == SDC_OK) {
        /* initialize - in case failing with error we want to return
         * buffer pointer NULL and buffer len 0 for all not explicitly specified buffers
         */
        if (SDC_OK != sdc_intern_check_init_dgst_tag_iv_output_buffer(digest_data, digest_len, &internal_digest_len, SDC_DGST_USE_DEFAULT, NULL)) {
            err = SDC_DGST_DATA_INVALID;
        }
    }

    if(err == SDC_OK) {
        if (internal_digest_len == SDC_DGST_USE_DEFAULT)
            internal_digest_len = desc->digest.dflt;
    }

    if (err == SDC_OK)
        err = sdc_dgst_common_check_digest(desc, internal_digest_len);

    if (err == SDC_OK) {
        if (internal_digest_len != 0) {
            internal_digest = malloc(internal_digest_len);
            if (!internal_digest)
                err = SDC_NO_MEM;
        }
    }

    if (err == SDC_OK)
        err = sdc_common_dgst_finalize(session, type, desc,
                                       internal_digest, internal_digest_len);

    if (err == SDC_OK) {
        *digest_data = internal_digest;
        *digest_len = internal_digest_len;
    } else {
        /* clean allocated memory */
        free(internal_digest);
    }

    if(err == SDC_OK) {
        err = sdc_intern_ops_sequence_ctl_set(session, SDC_OP_NO_CRYPTO, SDC_NO_OP);
    } else {
        /* If current operation is failed, reset the history of "operation" and "function" parameters
         * since doesn't support retry operation */
        tmp_err = sdc_intern_ops_sequence_ctl_set(session, SDC_OP_DGST, SDC_OP_ERROR);
        if(tmp_err != SDC_OK)
            err = tmp_err;
    }

    return err;
}
